@extends('layouts.app')

@section('title', 'Catálogo de Produtos - Suplementos Store')

@section('content')
<!-- Page Header -->
<section class="bg-gradient-to-r from-primary-700 to-primary-900 text-white py-12">
    <div class="container mx-auto px-4">
        <div class="max-w-4xl mx-auto text-center">
            <h1 class="text-3xl md:text-4xl font-bold mb-4">Catálogo de Produtos</h1>
            <p class="text-primary-100 text-lg">
                Encontre os melhores suplementos para atingir seus objetivos
            </p>
        </div>
    </div>
</section>

<!-- Main Content -->
<section class="py-12">
    <div class="container mx-auto px-4">
        <div class="flex flex-col lg:flex-row gap-8">
            <!-- Sidebar Filters -->
            <aside class="lg:w-1/4">
                <div class="bg-white rounded-xl shadow-md p-6 sticky top-24">
                    <div class="flex items-center justify-between mb-6">
                        <h2 class="text-lg font-bold text-gray-800">Filtros</h2>
                        <a href="{{ route('products.index') }}" class="text-sm text-primary-600 hover:underline">Limpar</a>
                    </div>

                    <!-- Categories Filter -->
                    <div class="mb-6">
                        <h3 class="font-semibold text-gray-700 mb-3">Categorias</h3>
                        <div class="space-y-2 max-h-48 overflow-y-auto">
                            @foreach($categories as $cat)
                                <label class="flex items-center cursor-pointer">
                                    <input type="checkbox" name="category" value="{{ $cat->slug }}"
                                           class="w-4 h-4 text-primary-600 rounded focus:ring-primary-500"
                                           @if(request('category') == $cat->slug) checked @endif
                                           onchange="applyFilters()">
                                    <span class="ml-2 text-gray-600">{{ $cat->name }}</span>
                                    <span class="ml-auto text-xs text-gray-400">({{ $cat->products_count }})</span>
                                </label>
                            @endforeach
                        </div>
                    </div>

                    <!-- Price Range Filter -->
                    <div class="mb-6">
                        <h3 class="font-semibold text-gray-700 mb-3">Faixa de Preço</h3>
                        <div class="space-y-3">
                            <div>
                                <label class="text-sm text-gray-600">Mínimo: R$ {{ number_format($minPrice ?? 0, 2, ',', '.') }}</label>
                                <input type="range" name="min_price" id="min_price"
                                       min="{{ $minPrice ?? 0 }}" max="{{ $maxPrice ?? 1000 }}"
                                       value="{{ request('min_price') ?? $minPrice ?? 0 }}"
                                       class="w-full h-2 bg-gray-200 rounded-lg appearance-none cursor-pointer"
                                       oninput="updatePriceLabels(); applyFilters()">
                            </div>
                            <div>
                                <label class="text-sm text-gray-600">Máximo: R$ {{ number_format($maxPrice ?? 1000, 2, ',', '.') }}</label>
                                <input type="range" name="max_price" id="max_price"
                                       min="{{ $minPrice ?? 0 }}" max="{{ $maxPrice ?? 1000 }}"
                                       value="{{ request('max_price') ?? $maxPrice ?? 1000 }}"
                                       class="w-full h-2 bg-gray-200 rounded-lg appearance-none cursor-pointer"
                                       oninput="updatePriceLabels(); applyFilters()">
                            </div>
                        </div>
                    </div>

                    <!-- Discount Filter -->
                    <div class="mb-6">
                        <label class="flex items-center cursor-pointer">
                            <input type="checkbox" name="with_discount" value="1"
                                   class="w-4 h-4 text-primary-600 rounded focus:ring-primary-500"
                                   @if(request('with_discount')) checked @endif
                                   onchange="applyFilters()">
                            <span class="ml-2 text-gray-600">Apenas com desconto</span>
                        </label>
                    </div>

                    <button onclick="clearFilters()" class="w-full py-2 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition">
                        Limpar Filtros
                    </button>
                </div>
            </aside>

            <!-- Products Grid -->
            <main class="lg:w-3/4">
                <!-- Sort Bar -->
                <div class="bg-white rounded-xl shadow-md p-4 mb-6 flex flex-col sm:flex-row items-center justify-between gap-4">
                    <div class="text-gray-600">
                        <span class="font-semibold">{{ $products->total() }}</span> produtos encontrados
                    </div>
                    <div class="flex items-center space-x-4">
                        <label class="text-gray-600 text-sm">Ordenar por:</label>
                        <select id="sort_by" onchange="applyFilters()" class="border border-gray-300 rounded-lg px-3 py-2 focus:outline-none focus:ring-2 focus:ring-primary-500">
                            <option value="name" {{ request('sort_by') == 'name' ? 'selected' : '' }}>Nome</option>
                            <option value="price" {{ request('sort_by') == 'price' ? 'selected' : '' }}>Preço</option>
                            <option value="featured" {{ request('sort_by') == 'featured' ? 'selected' : '' }}>Destaques</option>
                        </select>
                        <select id="sort_order" onchange="applyFilters()" class="border border-gray-300 rounded-lg px-3 py-2 focus:outline-none focus:ring-2 focus:ring-primary-500">
                            <option value="asc" {{ request('sort_order') == 'asc' ? 'selected' : '' }}>Crescente</option>
                            <option value="desc" {{ request('sort_order') == 'desc' ? 'selected' : '' }}>Decrescente</option>
                        </select>
                    </div>
                </div>

                <!-- Products -->
                @if($products->count() > 0)
                    <div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-6">
                        @foreach($products as $product)
                            @include('partials.product-card', ['product' => $product])
                        @endforeach
                    </div>

                    <!-- Pagination -->
                    @if($products->hasPages())
                        <div class="mt-8">
                            {{ $products->appends(request()->except('page'))->links() }}
                        </div>
                    @endif
                @else
                    <div class="bg-white rounded-xl shadow-md p-12 text-center">
                        <i class="fas fa-search text-gray-300 text-6xl mb-4"></i>
                        <h3 class="text-xl font-semibold text-gray-800 mb-2">Nenhum produto encontrado</h3>
                        <p class="text-gray-600 mb-4">Tente ajustar os filtros ou buscar por outro termo.</p>
                        <a href="{{ route('products.index') }}" class="inline-block px-6 py-2 bg-primary-600 text-white rounded-lg hover:bg-primary-700 transition">
                            Ver Todos os Produtos
                        </a>
                    </div>
                @endif
            </main>
        </div>
    </div>
</section>

@push('scripts')
<script>
function applyFilters() {
    const url = new URL(window.location);
    const categoryCheckboxes = document.querySelectorAll('input[name="category"]:checked');
    const minPrice = document.getElementById('min_price').value;
    const maxPrice = document.getElementById('max_price').value;
    const withDiscount = document.querySelector('input[name="with_discount"]')?.checked;
    const sortBy = document.getElementById('sort_by').value;
    const sortOrder = document.getElementById('sort_order').value;

    // Reset URL params
    url.searchParams.delete('category');
    url.searchParams.delete('min_price');
    url.searchParams.delete('max_price');
    url.searchParams.delete('with_discount');
    url.searchParams.delete('sort_by');
    url.searchParams.delete('sort_order');

    // Add categories
    categoryCheckboxes.forEach(checkbox => {
        url.searchParams.append('category', checkbox.value);
    });

    // Add price range
    url.searchParams.set('min_price', minPrice);
    url.searchParams.set('max_price', maxPrice);

    // Add discount
    if (withDiscount) {
        url.searchParams.set('with_discount', '1');
    }

    // Add sort
    url.searchParams.set('sort_by', sortBy);
    url.searchParams.set('sort_order', sortOrder);

    window.location = url.toString();
}

function clearFilters() {
    window.location = '{{ route('products.index') }}';
}

function updatePriceLabels() {
    // Visual feedback for price range
    const minPrice = document.getElementById('min_price').value;
    const maxPrice = document.getElementById('max_price').value;
    const minLabel = document.querySelector('label[for="min_price"]');
    const maxLabel = document.querySelector('label[for="max_price"]');

    if (minLabel) minLabel.textContent = 'Mínimo: R$ ' + parseFloat(minPrice).toFixed(2);
    if (maxLabel) maxLabel.textContent = 'Máximo: R$ ' + parseFloat(maxPrice).toFixed(2);
}
</script>
@endpush
@endsection
