<?php

namespace Database\Seeders;

use App\Models\Product;
use App\Models\ProductImage;
use App\Models\ProductFeature;
use App\Models\Category;
use Illuminate\Database\Seeder;
use Illuminate\Support\Str;

class ProductSeeder extends Seeder
{
    public function run(): void
    {
        $categories = Category::all()->keyBy('id');

        $products = [
            [
                'sku' => 'WHEY-001',
                'name' => 'Whey Protein Concentrado - Chocolate 900g',
                'slug' => 'whey-protein-concentrado-chocolate-900g',
                'description' => 'Whey Protein Concentrado é uma proteína de alta biodisponibilidade, ideal para quem busca ganho de massa muscular. Possui rápida absorção e é rico em aminoácidos essenciais, incluindo BCAAs.',
                'short_description' => 'Proteína de alta qualidade para crescimento muscular.',
                'category_id' => $categories->firstWhere('slug', 'proteinas')->id ?? 1,
                'brand' => 'Max Nutrition',
                'weight' => 900,
                'flavor' => 'Chocolate',
                'servings' => 30,
                'ingredients' => 'Whey Protein Concentrado, aromatizante, espessante, edulcorante.',
                'usage_instructions' => 'Misture 1 scoop (30g) em 200ml de água ou leite. Consumir após o treino.',
                'warnings' => 'Não consumir se tiver alergia a lactose ou proteínas do leite.',
                'price' => 149.90,
                'discount_price' => 119.90,
                'stock' => 50,
                'is_featured' => true,
                'features' => [
                    '24g de proteína por porção',
                    'Rico em BCAAs',
                    'Absorção rápida',
                    'Sem glúten',
                ],
            ],
            [
                'sku' => 'CREA-001',
                'name' => 'Creatina Monohidratada 300g',
                'slug' => 'creatina-monohidratada-300g',
                'description' => 'Creatina Monohidratada pura, o suplemento mais pesquisado e comprovado para aumentar força, potência e massa muscular. Ideal para atletas e praticantes de musculação.',
                'short_description' => 'Creatina pura para máxima força e performance.',
                'category_id' => $categories->firstWhere('slug', 'creatina')->id ?? 2,
                'brand' => 'Power Supplements',
                'weight' => 300,
                'flavor' => 'Neutro',
                'servings' => 60,
                'ingredients' => 'Creatina monohidratada 99%.',
                'usage_instructions' => 'Misture 3,5g em água ou suco. Consumir diariamente.',
                'warnings' => 'Manter hidratação adequada durante o uso.',
                'price' => 89.90,
                'discount_price' => 69.90,
                'stock' => 100,
                'is_featured' => true,
                'features' => [
                    'Creatina 99% pura',
                    'Aumenta força e potência',
                    'Melhora o desempenho',
                    '300 doses',
                ],
            ],
            [
                'sku' => 'BCAA-001',
                'name' => 'BCAA 2:1:1 - 240 Cápsulas',
                'slug' => 'bcaa-211-240-capsulas',
                'description' => 'BCAA em proporção ideal 2:1:1 de Leucina, Isoleucina e Valina. Essenciais para recuperação muscular e prevenção de catabolismo durante treinos intensos.',
                'short_description' => 'Aminoácidos de cadeia ramificada para recuperação.',
                'category_id' => $categories->firstWhere('slug', 'aminoacidos')->id ?? 3,
                'brand' => 'Fit Lab',
                'flavor' => null,
                'servings' => 80,
                'ingredients' => 'BCAA 2:1:1 (Leucina, Isoleucina, Valina), cápsula gelatinosa.',
                'usage_instructions' => 'Consumir 4 cápsulas antes do treino.',
                'warnings' => 'Não ultrapassar a dosagem recomendada.',
                'price' => 79.90,
                'discount_price' => null,
                'stock' => 75,
                'is_featured' => false,
                'features' => [
                    'Proporção 2:1:1',
                    'Previne catabolismo',
                    'Acelera recuperação',
                    'Fácil de consumir',
                ],
            ],
            [
                'sku' => 'TERM-001',
                'name' => 'Termogênico Pro Burn - 120 Cápsulas',
                'slug' => 'termogenico-pro-burn-120-capsulas',
                'description' => 'Termogênico avançado com cafeína anidra, chá verde e guaraná. Formulação exclusiva para acelerar o metabolismo e aumentar a queima de gordura durante o dia.',
                'short_description' => 'Termogênico potente para queima de gordura.',
                'category_id' => $categories->firstWhere('slug', 'termogenicos')->id ?? 4,
                'brand' => 'Burn Nutrition',
                'flavor' => null,
                'servings' => 60,
                'ingredients' => 'Cafeína Anidra, Extrato de Chá Verde, Guaraná, L-Carnitina, Pimenta Caiena.',
                'usage_instructions' => 'Consumir 2 cápsulas pela manhã e 2 à tarde.',
                'warnings' => 'Contém cafeína. Não consumir após 18h.',
                'price' => 129.90,
                'discount_price' => 99.90,
                'stock' => 60,
                'is_featured' => true,
                'features' => [
                    'Acelera metabolismo',
                    'Aumenta energia',
                    'Inibe apetite',
                    'Fórmula avançada',
                ],
            ],
            [
                'sku' => 'MULT-001',
                'name' => 'Multivitamínico Complete - 60 Cápsulas',
                'slug' => 'multivitaminico-complete-60-capsulas',
                'description' => 'Multivitamínico completo com 25 vitaminas e minerais essenciais. Ideal para garantir a ingestão diária de micronutrientes e manter a saúde e imunidade em dia.',
                'short_description' => 'Multivitamínico completo para saúde e imunidade.',
                'category_id' => $categories->firstWhere('slug', 'vitaminas-minerais')->id ?? 5,
                'brand' => 'Health Plus',
                'flavor' => null,
                'servings' => 60,
                'ingredients' => 'Vitamina A, C, D, E, Complexo B, Zinco, Magnésio, Cálcio e mais.',
                'usage_instructions' => 'Consumir 1 cápsula diária com café da manhã.',
                'warnings' => 'Não substitui uma alimentação equilibrada.',
                'price' => 59.90,
                'discount_price' => null,
                'stock' => 200,
                'is_featured' => false,
                'features' => [
                    '25 vitaminas e minerais',
                    'Fortalece imunidade',
                    '60 dias de suplementação',
                    'Fácil absorção',
                ],
            ],
            [
                'sku' => 'PRE-001',
                'name' => 'Pré-Treino Ignite - 300g',
                'slug' => 'pre-treino-ignite-300g',
                'description' => 'Pré-treino potente com beta-alanina, citrulina, cafeína e taurina. Formulação explosiva para máxima energia, foco e pump muscular durante os treinos.',
                'short_description' => 'Explosão de energia e foco para seus treinos.',
                'category_id' => $categories->firstWhere('slug', 'pre-treino')->id ?? 6,
                'brand' => 'Ignite Supplements',
                'weight' => 300,
                'flavor' => 'Frutas Vermelhas',
                'servings' => 30,
                'ingredients' => 'Beta-Alanina, L-Citrulina, Cafeína Anidra, Taurina, Tirosina.',
                'usage_instructions' => 'Misturar 1 scoop (10g) em 200ml de água 30min antes do treino.',
                'warnings' => 'Contém cafeína. Não usar se for sensível.',
                'price' => 139.90,
                'discount_price' => 109.90,
                'stock' => 80,
                'is_featured' => true,
                'features' => [
                    'Energia explosiva',
                    'Foco mental',
                    'Pump muscular',
                    'Sem açúcar',
                ],
            ],
            [
                'sku' => 'WHEY-ISO-001',
                'name' => 'Whey Protein Isolado - Morango 900g',
                'slug' => 'whey-protein-isolado-morango-900g',
                'description' => 'Whey Protein Isolado, a forma mais pura de proteína do soro do leite. Zero gordura e zero carboidratos, com 27g de proteína por porção. Perfeito para dieta de cutting.',
                'short_description' => 'Proteína isolada ultra-pura, zero gordura.',
                'category_id' => $categories->firstWhere('slug', 'proteinas')->id ?? 1,
                'brand' => 'Max Nutrition',
                'weight' => 900,
                'flavor' => 'Morango',
                'servings' => 30,
                'ingredients' => 'Whey Protein Isolado, aromatizante, edulcorante.',
                'usage_instructions' => 'Misture 1 scoop (30g) em 200ml de água. Consumir após treino.',
                'warnings' => 'Não consumir se tiver alergia a proteínas do leite.',
                'price' => 199.90,
                'discount_price' => 169.90,
                'stock' => 40,
                'is_featured' => true,
                'features' => [
                    '27g de proteína por dose',
                    'Zero gordura',
                    'Zero carboidratos',
                    'Absorção ultrarrápida',
                ],
            ],
            [
                'sku' => 'ZMA-001',
                'name' => 'ZMA - Zinco Magnésio Vitamina B6 - 90 Cápsulas',
                'slug' => 'zma-zinco-magnesio-vitamina-b6-90-capsulas',
                'description' => 'ZMA é a combinação perfeita de Zinco, Magnésio e Vitamina B6. Melhora a qualidade do sono, recuperação muscular e níveis de testosterona natural.',
                'short_description' => 'Sono profundo e recuperação otimizada.',
                'category_id' => $categories->firstWhere('slug', 'vitaminas-minerais')->id ?? 5,
                'brand' => 'Night Recovery',
                'flavor' => null,
                'servings' => 90,
                'ingredients' => 'Zinco (como monometionina), Magnésio (aspartato), Vitamina B6.',
                'usage_instructions' => 'Consumir 3 cápsulas antes de dormir em estômago vazio.',
                'warnings' => 'Não usar com cálcio no mesmo horário.',
                'price' => 69.90,
                'discount_price' => null,
                'stock' => 120,
                'is_featured' => false,
                'features' => [
                    'Melhora sono',
                    'Aumenta testosterona',
                    'Recuperação noturna',
                    '90 dias de uso',
                ],
            ],
            [
                'sku' => 'OMEGA-001',
                'name' => 'Ômega 3 Fish Oil - 120 Cápsulas',
                'slug' => 'omega-3-fish-oil-120-capsulas',
                'description' => 'Ômega 3 de alta concentração com 1000mg de óleo de peixe por cápsula. Rico em EPA e DHA, essenciais para saúde cardiovascular, cerebral e das articulações.',
                'short_description' => 'Ômega 3 de alta potência para saúde integral.',
                'category_id' => $categories->firstWhere('slug', 'vitaminas-minerais')->id ?? 5,
                'brand' => 'Ocean Health',
                'flavor' => null,
                'servings' => 120,
                'ingredients' => 'Óleo de peixe (EPA 180mg, DHA 120mg), cápsula de gelatina.',
                'usage_instructions' => 'Consumir 2 cápsulas com almoço e 2 com jantar.',
                'warnings' => 'Consulte médico se usar anticoagulantes.',
                'price' => 79.90,
                'discount_price' => 59.90,
                'stock' => 90,
                'is_featured' => false,
                'features' => [
                    'Alta concentração',
                    'Saúde cardiovascular',
                    'Função cerebral',
                    'Articulações saudáveis',
                ],
            ],
            [
                'sku' => 'CASEINA-001',
                'name' => 'Caseína Micelar - Chocolate 1kg',
                'slug' => 'caseina-micelar-chocolate-1kg',
                'description' => 'Caseína Micelar é uma proteína de absorção lenta, ideal para consumo antes de dormir. Fornece aminoácidos por até 7 horas, prevenindo catabolismo noturno.',
                'short_description' => 'Proteína de absorção lenta para noite.',
                'category_id' => $categories->firstWhere('slug', 'proteinas')->id ?? 1,
                'brand' => 'Night Protein',
                'weight' => 1000,
                'flavor' => 'Chocolate',
                'servings' => 33,
                'ingredients' => 'Caseína Micelar, aromatizante, espessante, edulcorante.',
                'usage_instructions' => 'Misture 1 scoop (30g) em 250ml de leite antes de dormir.',
                'warnings' => 'Contém lactose.',
                'price' => 169.90,
                'discount_price' => null,
                'stock' => 55,
                'is_featured' => false,
                'features' => [
                    'Absorção prolongada',
                    'Previne catabolismo',
                    'Saciedade noturna',
                    'Sabor cremoso',
                ],
            ],
        ];

        foreach ($products as $productData) {
            $features = $productData['features'] ?? [];
            unset($productData['features']);

            $product = Product::create($productData);

            // Calculate discount percentage
            if ($product->discount_price && $product->discount_price < $product->price) {
                $product->discount_percentage = round((($product->price - $product->discount_price) / $product->price) * 100);
                $product->save();
            }

            // Create features
            foreach ($features as $index => $feature) {
                ProductFeature::create([
                    'product_id' => $product->id,
                    'feature' => $feature,
                    'order' => $index,
                ]);
            }

            // Create placeholder image URL
            $productImages[] = [
                'product_id' => $product->id,
                'url' => "https://placehold.co/400x400/1e40af/ffffff?text=" . urlencode(strtoupper(substr($product->name, 0, 15))),
                'alt_text' => $product->name,
                'is_main' => true,
                'order' => 0,
            ];
        }

        // Insert all images
        foreach ($productImages as $image) {
            ProductImage::create($image);
        }
    }
}
