<?php

namespace App\Services;

use App\Models\Order;
use App\Models\Setting;
use Illuminate\Support\Facades\Http;

class AsaasService
{
    protected $apiKey;
    protected $sandbox;
    protected $baseUrl;

    public function __construct()
    {
        $this->apiKey = Setting::getAsaasApiKey();
        $this->sandbox = Setting::getAsaasIsSandbox();
        $this->baseUrl = $this->sandbox
            ? 'https://sandbox.asaas.com/api/v3/'
            : 'https://www.asaas.com/api/v3/';
    }

    /**
     * Criar um cliente no Asaas
     */
    public function createCustomer(array $data)
    {
        try {
            $response = Http::withHeaders([
                'access_token' => $this->apiKey,
                'Content-Type' => 'application/json',
            ])->post($this->baseUrl . 'customers', [
                'name' => $data['name'],
                'email' => $data['email'],
                'phone' => $this->formatPhone($data['phone']),
                'mobilePhone' => $this->formatPhone($data['phone']),
                'cpfCnpj' => preg_replace('/[^0-9]/', '', $data['cpf'] ?? ''),
                'postalCode' => preg_replace('/[^0-9]/', '', $data['zipcode']),
                'address' => $data['address'],
                'addressNumber' => $data['number'],
                'complement' => $data['complement'] ?? '',
                'province' => $data['district'],
                'city' => $data['city'],
                'state' => $data['state'],
                'country' => 'BR',
                'externalReference' => $data['external_reference'] ?? null,
            ]);

            if ($response->successful()) {
                return $response->json();
            }

            throw new \Exception('Erro ao criar cliente: ' . $response->body());
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Criar pagamento PIX
     */
    public function createPixPayment(Order $order, ?string $customerId = null)
    {
        try {
            $data = [
                'customer' => $customerId ?? $this->createCustomerFromOrder($order),
                'billingType' => 'PIX',
                'value' => $this->convertCents($order->total),
                'dueDate' => now()->addDays(3)->format('Y-m-d'),
                'description' => "Pedido #{$order->order_number}",
                'externalReference' => $order->order_number,
                'postalService' => 'Sedex',
            ];

            $response = Http::withHeaders([
                'access_token' => $this->apiKey,
                'Content-Type' => 'application/json',
            ])->post($this->baseUrl . 'payments', $data);

            if ($response->successful()) {
                $payment = $response->json();

                // Salvar dados do PIX no pedido
                $order->update([
                    'gateway_transaction_id' => $payment['id'],
                    'gateway_payment_link' => $payment['invoiceUrl'] ?? null,
                    'pix_qr_code' => $payment['encodedImage'] ?? null,
                    'pix_copy_paste' => $payment['payload'] ?? null,
                ]);

                return $payment;
            }

            throw new \Exception('Erro ao criar pagamento PIX: ' . $response->body());
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Criar pagamento com Boleto
     */
    public function createBoletoPayment(Order $order, ?string $customerId = null)
    {
        try {
            $data = [
                'customer' => $customerId ?? $this->createCustomerFromOrder($order),
                'billingType' => 'BOLETO',
                'value' => $this->convertCents($order->total),
                'dueDate' => now()->addDays(3)->format('Y-m-d'),
                'description' => "Pedido #{$order->order_number}",
                'externalReference' => $order->order_number,
                'postalService' => 'Sedex',
            ];

            $response = Http::withHeaders([
                'access_token' => $this->apiKey,
                'Content-Type' => 'application/json',
            ])->post($this->baseUrl . 'payments', $data);

            if ($response->successful()) {
                $payment = $response->json();

                // Salvar dados do boleto no pedido
                $order->update([
                    'gateway_transaction_id' => $payment['id'],
                    'gateway_payment_link' => $payment['invoiceUrl'] ?? null,
                    'boleto_url' => $payment['invoiceUrl'] ?? null,
                    'boleto_barcode' => $payment['barcode'] ?? null,
                ]);

                return $payment;
            }

            throw new \Exception('Erro ao criar pagamento com Boleto: ' . $response->body());
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Criar pagamento com Cartão de Crédito
     */
    public function createCreditCardPayment(Order $order, array $cardData, ?string $customerId = null)
    {
        try {
            $data = [
                'customer' => $customerId ?? $this->createCustomerFromOrder($order),
                'billingType' => 'CREDIT_CARD',
                'value' => $this->convertCents($order->total),
                'dueDate' => now()->format('Y-m-d'),
                'description' => "Pedido #{$order->order_number}",
                'externalReference' => $order->order_number,
                'creditCard' => [
                    'creditCardNumber' => preg_replace('/[^0-9]/', '', $cardData['number']),
                    'creditCardHolderName' => $cardData['holder_name'],
                    'creditCardExpirationMonth' => $cardData['expiration_month'],
                    'creditCardExpirationYear' => $cardData['expiration_year'],
                    'creditCardCvv' => $cardData['cvv'],
                ],
                'creditCardHolderInfo' => [
                    'name' => $order->customer_name,
                    'email' => $order->customer_email,
                    'cpfCnpj' => preg_replace('/[^0-9]/', '', $order->customer_cpf ?? ''),
                    'postalCode' => preg_replace('/[^0-9]/', '', $order->shipping_zipcode),
                    'addressNumber' => $order->shipping_number,
                    'addressComplement' => $order->shipping_complement ?? '',
                    'mobilePhone' => $this->formatPhone($order->customer_phone),
                ],
                'remoteIp' => request()->ip(),
            ];

            // Se houver parcelamento
            if (!empty($cardData['installments'])) {
                $data['installmentCount'] = $cardData['installments'];
                $data['installmentValue'] = $this->convertCents($order->total / $cardData['installments']);
            }

            $response = Http::withHeaders([
                'access_token' => $this->apiKey,
                'Content-Type' => 'application/json',
            ])->post($this->baseUrl . 'payments', $data);

            if ($response->successful()) {
                $payment = $response->json();

                // Salvar dados da transação no pedido
                $order->update([
                    'gateway_transaction_id' => $payment['id'],
                ]);

                return $payment;
            }

            throw new \Exception('Erro ao criar pagamento com Cartão: ' . $response->body());
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Consultar status de um pagamento
     */
    public function getPaymentStatus(string $paymentId)
    {
        try {
            $response = Http::withHeaders([
                'access_token' => $this->apiKey,
            ])->get($this->baseUrl . 'payments/' . $paymentId);

            if ($response->successful()) {
                return $response->json();
            }

            return null;
        } catch (\Exception $e) {
            return null;
        }
    }

    /**
     * Atualizar status do pedido baseado no webhook do Asaas
     */
    public function updateOrderStatusFromWebhook(array $webhookData)
    {
        try {
            $payment = $this->getPaymentStatus($webhookData['payment']['id']);

            if (!$payment) {
                return false;
            }

            $order = Order::where('order_number', $payment['externalReference'])->first();

            if (!$order) {
                return false;
            }

            // Atualizar status do pedido baseado no status do pagamento
            $statusMap = [
                'PENDING' => Order::PAYMENT_PENDING,
                'CONFIRMED' => Order::PAYMENT_PAID,
                'RECEIVED' => Order::PAYMENT_PAID,
                'OVERDUE' => Order::PAYMENT_PENDING,
                'CANCELED' => Order::PAYMENT_CANCELLED,
            ];

            if (isset($statusMap[$payment['status']])) {
                $order->update([
                    'payment_status' => $statusMap[$payment['status']],
                    'status' => $statusMap[$payment['status']] === Order::PAYMENT_PAID ? Order::STATUS_CONFIRMED : Order::STATUS_PENDING,
                ]);
            }

            return true;
        } catch (\Exception $e) {
            return false;
        }
    }

    /**
     * Criar cliente a partir do pedido
     */
    protected function createCustomerFromOrder(Order $order)
    {
        $customerData = [
            'name' => $order->customer_name,
            'email' => $order->customer_email,
            'phone' => $order->customer_phone,
            'cpf' => $order->customer_cpf ?? '',
            'zipcode' => $order->shipping_zipcode,
            'address' => $order->shipping_address,
            'number' => $order->shipping_number,
            'complement' => $order->shipping_complement,
            'district' => $order->shipping_district,
            'city' => $order->shipping_city,
            'state' => $order->shipping_state,
            'external_reference' => $order->customer_email,
        ];

        $customer = $this->createCustomer($customerData);

        return $customer['id'];
    }

    /**
     * Formatar telefone para o formato do Asaas
     */
    protected function formatPhone(string $phone): string
    {
        $phone = preg_replace('/[^0-9]/', '', $phone);

        // Adicionar DDD 11 se não tiver
        if (strlen($phone) === 10) {
            $phone = substr($phone, 0, 2) . '9' . substr($phone, 2);
        }

        return '+' . $phone;
    }

    /**
     * Converter valor para centavos
     */
    protected function convertCents(float $value): int
    {
        return (int) round($value * 100);
    }

    /**
     * Verificar se PIX está habilitado
     */
    public function isPixEnabled(): bool
    {
        return Setting::getAsaasPixEnabled();
    }

    /**
     * Verificar se Cartão de Crédito está habilitado
     */
    public function isCreditCardEnabled(): bool
    {
        return Setting::getAsaasCreditCardEnabled();
    }

    /**
     * Verificar se Boleto está habilitado
     */
    public function isBoletoEnabled(): bool
    {
        return Setting::getAsaasBoletoEnabled();
    }

    /**
     * Obter o valor com desconto PIX
     */
    public function getPixPrice(float $originalPrice): float
    {
        $discount = Setting::get('pix_discount_percentage', 0);
        return $originalPrice * (1 - ($discount / 100));
    }
}
