<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Cache;

class Setting extends Model
{
    protected $fillable = [
        'key',
        'value',
        'type',
        'group',
    ];

    public static function get(string $key, $default = null)
    {
        return Cache::remember("setting.{$key}", 3600, function () use ($key, $default) {
            $setting = self::where('key', $key)->first();

            if (!$setting) {
                return $default;
            }

            return match ($setting->type) {
                'boolean' => filter_var($setting->value, FILTER_VALIDATE_BOOLEAN),
                'number' => is_numeric($setting->value) ? (float) $setting->value : $default,
                'json' => json_decode($setting->value, true),
                default => $setting->value,
            };
        });
    }

    public static function set(string $key, $value, string $type = 'text', string $group = 'general'): void
    {
        self::updateOrCreate(
            ['key' => $key],
            [
                'value' => is_array($value) ? json_encode($value) : $value,
                'type' => $type,
                'group' => $group,
            ]
        );

        Cache::forget("setting.{$key}");
    }

    public static function getGroup(string $group): array
    {
        return self::where('group', $group)
            ->get()
            ->pluck('value', 'key')
            ->toArray();
    }

    // Configurações comuns
    public static function getStoreName(): string
    {
        return self::get('store_name', 'Minha Loja');
    }

    public static function getStoreEmail(): string
    {
        return self::get('store_email', 'contato@minhaloja.com');
    }

    public static function getStorePhone(): string
    {
        return self::get('store_phone', '');
    }

    // Asaas
    public static function getAsaasApiKey(): ?string
    {
        return self::get('asaas_api_key');
    }

    public static function getAsaasIsSandbox(): bool
    {
        return self::get('asaas_sandbox', true);
    }

    public static function getAsaasPixEnabled(): bool
    {
        return self::get('asaas_pix_enabled', true);
    }

    public static function getAsaasCreditCardEnabled(): bool
    {
        return self::get('asaas_credit_card_enabled', true);
    }

    public static function getAsaasBoletoEnabled(): bool
    {
        return self::get('asaas_boleto_enabled', true);
    }

    // Mercado Pago
    public static function getMercadoPagoAccessToken(): ?string
    {
        return self::get('mercadopago_access_token');
    }

    public static function getMercadoPagoIsSandbox(): bool
    {
        return self::get('mercadopago_sandbox', true);
    }

    // Frete
    public static function getFreeShippingThreshold(): ?float
    {
        return self::get('free_shipping_threshold');
    }

    public static function getDefaultShippingCost(): float
    {
        return self::get('default_shipping_cost', 0);
    }
}
