<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Product extends Model
{
    protected $fillable = [
        'sku',
        'name',
        'slug',
        'description',
        'short_description',
        'category_id',
        'brand',
        'weight',
        'flavor',
        'servings',
        'ingredients',
        'usage_instructions',
        'warnings',
        'price',
        'discount_price',
        'discount_percentage',
        'stock',
        'is_featured',
        'is_active',
        'meta_title',
        'meta_description',
        'video_url',
        'video_type',
        'max_images',
    ];

    protected $casts = [
        'price' => 'decimal:2',
        'discount_price' => 'decimal:2',
        'weight' => 'decimal:2',
        'is_featured' => 'boolean',
        'is_active' => 'boolean',
    ];

    public function category(): BelongsTo
    {
        return $this->belongsTo(Category::class);
    }

    public function images(): HasMany
    {
        return $this->hasMany(ProductImage::class)->orderBy('order');
    }

    public function mainImage(): HasMany
    {
        return $this->hasMany(ProductImage::class)->where('is_main', true);
    }

    public function features(): HasMany
    {
        return $this->hasMany(ProductFeature::class)->orderBy('order');
    }

    public function variants()
    {
        return $this->belongsToMany(Variant::class, 'product_variants')
            ->using(ProductVariant::class)
            ->withPivot('id', 'value', 'price_adjustment', 'sku', 'stock', 'is_active', 'sort_order')
            ->withTimestamps();
    }

    public function productVariants()
    {
        return $this->hasMany(ProductVariant::class);
    }

    public function combos()
    {
        return $this->belongsToMany(Combo::class, 'combo_items')
            ->withPivot('quantity')
            ->withTimestamps();
    }

    public function getVariantsByTypeAttribute(): array
    {
        $variants = [];
        foreach ($this->variants as $variant) {
            if (!isset($variants[$variant->type])) {
                $variants[$variant->type] = [
                    'name' => $variant->name,
                    'values' => [],
                ];
            }
            $variants[$variant->type]['values'][] = [
                'id' => $variant->pivot->id,
                'value' => $variant->pivot->value,
                'price_adjustment' => $variant->pivot->price_adjustment,
                'sku' => $variant->pivot->sku,
                'stock' => $variant->pivot->stock,
            ];
        }
        return $variants;
    }

    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeFeatured($query)
    {
        return $query->where('is_featured', true);
    }

    public function scopeWithDiscount($query)
    {
        return $query->whereNotNull('discount_price');
    }

    public function getFinalPriceAttribute(): float
    {
        return $this->discount_price ?? $this->price;
    }

    public function getHasDiscountAttribute(): bool
    {
        return $this->discount_price !== null && $this->discount_price < $this->price;
    }

    public function getDiscountPercentageCalculatedAttribute(): ?int
    {
        if (!$this->has_discount) {
            return null;
        }
        return round((($this->price - $this->discount_price) / $this->price) * 100);
    }
}
