<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Order extends Model
{
    use SoftDeletes;

    protected $fillable = [
        'order_number',
        'user_id',
        'customer_name',
        'customer_email',
        'customer_phone',
        'customer_cpf',
        'shipping_zipcode',
        'shipping_address',
        'shipping_number',
        'shipping_complement',
        'shipping_district',
        'shipping_city',
        'shipping_state',
        'subtotal',
        'shipping_cost',
        'discount',
        'total',
        'payment_method',
        'payment_status',
        'gateway',
        'gateway_transaction_id',
        'gateway_payment_link',
        'pix_qr_code',
        'pix_copy_paste',
        'boleto_url',
        'boleto_barcode',
        'status',
        'tracking_code',
        'coupon_id',
        'coupon_code',
        'coupon_discount',
        'installments',
        'installment_value',
        'admin_notes',
    ];

    protected $casts = [
        'subtotal' => 'decimal:2',
        'shipping_cost' => 'decimal:2',
        'discount' => 'decimal:2',
        'total' => 'decimal:2',
    ];

    // Status do pedido
    const STATUS_PENDING = 'pending';
    const STATUS_CONFIRMED = 'confirmed';
    const STATUS_PROCESSING = 'processing';
    const STATUS_SHIPPED = 'shipped';
    const STATUS_DELIVERED = 'delivered';
    const STATUS_CANCELLED = 'cancelled';

    // Status de pagamento
    const PAYMENT_PENDING = 'pending';
    const PAYMENT_PAID = 'paid';
    const PAYMENT_CANCELLED = 'cancelled';
    const PAYMENT_REFUNDED = 'refunded';

    // Métodos de pagamento
    const PAYMENT_PIX = 'pix';
    const PAYMENT_CREDIT_CARD = 'credit_card';
    const PAYMENT_BOLETO = 'boleto';

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function items()
    {
        return $this->hasMany(OrderItem::class);
    }

    public function coupon()
    {
        return $this->belongsTo(Coupon::class);
    }

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($order) {
            if (empty($order->order_number)) {
                $order->order_number = 'ORD-' . strtoupper(uniqid());
            }
        });
    }

    public function getStatusLabelAttribute(): string
    {
        return [
            self::STATUS_PENDING => 'Pendente',
            self::STATUS_CONFIRMED => 'Confirmado',
            self::STATUS_PROCESSING => 'Processando',
            self::STATUS_SHIPPED => 'Enviado',
            self::STATUS_DELIVERED => 'Entregue',
            self::STATUS_CANCELLED => 'Cancelado',
        ][$this->status] ?? 'Desconhecido';
    }

    public function getPaymentStatusLabelAttribute(): string
    {
        return [
            self::PAYMENT_PENDING => 'Pendente',
            self::PAYMENT_PAID => 'Pago',
            self::PAYMENT_CANCELLED => 'Cancelado',
            self::PAYMENT_REFUNDED => 'Reembolsado',
        ][$this->payment_status] ?? 'Desconhecido';
    }

    public function getPaymentMethodLabelAttribute(): string
    {
        return [
            self::PAYMENT_PIX => 'PIX',
            self::PAYMENT_CREDIT_CARD => 'Cartão de Crédito',
            self::PAYMENT_BOLETO => 'Boleto',
        ][$this->payment_method] ?? 'Desconhecido';
    }

    public function getFullAddressAttribute(): string
    {
        return "{$this->shipping_address}, {$this->shipping_number}" .
            ($this->shipping_complement ? " - {$this->shipping_complement}" : '') .
            " - {$this->shipping_district}/{$this->shipping_city} - {$this->shipping_state}";
    }

    public function scopeByStatus($query, $status)
    {
        return $query->where('status', $status);
    }

    public function scopeByPaymentStatus($query, $status)
    {
        return $query->where('payment_status', $status);
    }

    public function scopeLatest($query)
    {
        return $query->orderBy('created_at', 'desc');
    }
}
