<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;

class Coupon extends Model
{
    protected $fillable = [
        'code',
        'type',
        'value',
        'min_purchase',
        'max_uses',
        'used_count',
        'valid_from',
        'valid_until',
        'is_active',
    ];

    protected $casts = [
        'value' => 'decimal:2',
        'min_purchase' => 'decimal:2',
        'is_active' => 'boolean',
        'valid_from' => 'date',
        'valid_until' => 'date',
    ];

    const TYPE_PERCENTAGE = 'percentage';
    const TYPE_FIXED = 'fixed';

    public function orders()
    {
        return $this->hasMany(Order::class);
    }

    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeValid($query)
    {
        return $query->where(function ($q) {
            $q->whereNull('valid_from')
                ->orWhere('valid_from', '<=', now());
        })->where(function ($q) {
            $q->whereNull('valid_until')
                ->orWhere('valid_until', '>=', now());
        });
    }

    public function scopeAvailable($query)
    {
        return $query->where(function ($q) {
            $q->whereNull('max_uses')
                ->whereColumn('used_count', '<', 'max_uses');
        });
    }

    public function isValid(): bool
    {
        if (!$this->is_active) {
            return false;
        }

        if ($this->valid_from && $this->valid_from->isFuture()) {
            return false;
        }

        if ($this->valid_until && $this->valid_until->isPast()) {
            return false;
        }

        if ($this->max_uses && $this->used_count >= $this->max_uses) {
            return false;
        }

        return true;
    }

    public function calculateDiscount(float $subtotal): float
    {
        if (!$this->isValid()) {
            return 0;
        }

        if ($this->min_purchase && $subtotal < $this->min_purchase) {
            return 0;
        }

        return $this->type === self::TYPE_PERCENTAGE
            ? $subtotal * ($this->value / 100)
            : $this->value;
    }

    public function getTypeLabelAttribute(): string
    {
        return $this->type === self::TYPE_PERCENTAGE ? 'Porcentagem' : 'Valor Fixo';
    }

    public function getDisplayValueAttribute(): string
    {
        return $this->type === self::TYPE_PERCENTAGE
            ? $this->value . '%'
            : 'R$ ' . number_format($this->value, 2, ',', '.');
    }

    public function incrementUsedCount(): void
    {
        $this->increment('used_count');
    }
}
