<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;

class Combo extends Model
{
    protected $fillable = [
        'name',
        'slug',
        'description',
        'original_price',
        'price',
        'pix_price',
        'max_installments',
        'min_installment_value',
        'is_active',
        'is_featured',
        'free_shipping_threshold',
        'image',
        'video_url',
        'gifts',
    ];

    protected $casts = [
        'original_price' => 'decimal:2',
        'price' => 'decimal:2',
        'pix_price' => 'decimal:2',
        'min_installment_value' => 'decimal:2',
        'is_active' => 'boolean',
        'is_featured' => 'boolean',
        'free_shipping_threshold' => 'integer',
        'gifts' => 'array',
    ];

    public function items()
    {
        return $this->hasMany(ComboItem::class);
    }

    public function products()
    {
        return $this->belongsToMany(Product::class, 'combo_items')
            ->withPivot('quantity')
            ->withTimestamps();
    }

    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeFeatured($query)
    {
        return $query->where('is_featured', true);
    }

    public function getDiscountPercentageAttribute(): float
    {
        if ($this->original_price > 0) {
            return round((($this->original_price - $this->price) / $this->original_price) * 100);
        }
        return 0;
    }

    public function getInstallmentValueAttribute(): float
    {
        if ($this->max_installments > 0) {
            return ceil($this->price / $this->max_installments * 100) / 100;
        }
        return $this->price;
    }

    public function hasFreeShipping(float $cartTotal): bool
    {
        return $this->free_shipping_threshold && $cartTotal >= $this->free_shipping_threshold;
    }
}
