<?php

namespace App\Http\Controllers;

use App\Models\Product;
use App\Models\Category;
use App\Models\Combo;
use App\Models\Variant;
use Illuminate\Http\Request;

class ProductController extends Controller
{
    public function index(Request $request)
    {
        $query = Product::active()->with(['category', 'images']);

        // Filter by category
        if ($request->has('category') && $request->category) {
            $query->whereHas('category', function ($q) use ($request) {
                $q->where('slug', $request->category);
            });
        }

        // Filter by search
        if ($request->has('search') && $request->search) {
            $query->where(function ($q) use ($request) {
                $q->where('name', 'like', '%' . $request->search . '%')
                    ->orWhere('description', 'like', '%' . $request->search . '%')
                    ->orWhere('short_description', 'like', '%' . $request->search . '%');
            });
        }

        // Filter by price range
        if ($request->has('min_price') && $request->min_price) {
            $query->where('final_price', '>=', $request->min_price);
        }
        if ($request->has('max_price') && $request->max_price) {
            $query->where('final_price', '<=', $request->max_price);
        }

        // Filter only with discount
        if ($request->has('with_discount') && $request->with_discount) {
            $query->withDiscount();
        }

        // Sorting
        $sortBy = $request->get('sort_by', 'name');
        $sortOrder = $request->get('sort_order', 'asc');

        switch ($sortBy) {
            case 'price':
                $query->orderBy('final_price', $sortOrder);
                break;
            case 'name':
            default:
                $query->orderBy('name', $sortOrder);
                break;
            case 'featured':
                $query->orderBy('is_featured', 'desc');
                break;
            case 'discount':
                $query->orderByRaw('discount_percentage DESC NULLS LAST');
                break;
        }

        $products = $query->paginate(12)->withQueryString();

        $categories = Category::active()->root()->withCount('products')->get();

        $minPrice = Product::active()->min('final_price');
        $maxPrice = Product::active()->max('final_price');

        return view('products.index', compact(
            'products',
            'categories',
            'minPrice',
            'maxPrice'
        ));
    }

    public function show($slug)
    {
        $product = Product::active()
            ->with(['category', 'categories', 'images', 'features', 'variants.variant'])
            ->where('slug', $slug)
            ->firstOrFail();

        // Variantes disponíveis agrupadas por tipo
        $availableVariants = Variant::active()
            ->orderBy('type')
            ->orderBy('sort_order')
            ->get()
            ->groupBy('type');

        // Produtos relacionados (mesma categoria)
        $relatedProducts = Product::active()
            ->whereHas('categories', function ($q) use ($product) {
                $q->whereIn('categories.id', $product->categories->pluck('id'));
            })
            ->where('id', '!=', $product->id)
            ->with(['category', 'images', 'categories'])
            ->limit(4)
            ->get();

        // Combos que incluem este produto (kits)
        $combos = Combo::with(['items.product'])
            ->whereHas('items', function ($q) use ($product) {
                $q->where('product_id', $product->id);
            })
            ->active()
            ->get();

        return view('products.show', compact('product', 'relatedProducts', 'combos', 'availableVariants'));
    }
}
