<?php

namespace App\Http\Controllers;

use App\Models\Order;
use App\Models\OrderItem;
use App\Models\Product;
use App\Models\Combo;
use App\Models\Setting;
use App\Services\AsaasService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Session;

class CheckoutController extends Controller
{
    protected $asaasService;

    public function __construct(AsaasService $asaasService)
    {
        $this->asaasService = $asaasService;
    }

    /**
     * Obter o carrinho da sessão
     */
    protected function getCart()
    {
        return Session::get('cart', [
            'items' => [],
            'coupon' => null,
        ]);
    }

    /**
     * Salvar o carrinho na sessão
     */
    protected function saveCart(array $cart): void
    {
        Session::put('cart', $cart);
    }

    /**
     * Calcular totais do carrinho
     */
    protected function calculateTotals(array &$cart): void
    {
        $cart['subtotal'] = collect($cart['items'])->sum(function ($item) {
            return ($item['discount_price'] ?? $item['price']) * $item['quantity'];
        });

        // Desconto do cupom
        $cart['discount'] = 0;
        if ($cart['coupon']) {
            if ($cart['coupon']['type'] === 'percentage') {
                $cart['discount'] = $cart['subtotal'] * ($cart['coupon']['value'] / 100);
            } else {
                $cart['discount'] = min($cart['coupon']['value'], $cart['subtotal']);
            }
        }

        // Frete
        $cart['shipping_cost'] = 0;
        $cart['free_shipping'] = false;

        $freeShippingThreshold = Setting::get('free_shipping_threshold');
        if ($freeShippingThreshold && $cart['subtotal'] >= $freeShippingThreshold) {
            $cart['free_shipping'] = true;
        } else {
            $cart['shipping_cost'] = Setting::get('default_shipping_cost', 0);
        }

        // Total
        $cart['total'] = $cart['subtotal'] - $cart['discount'] + $cart['shipping_cost'];

        // Total PIX (com desconto adicional)
        $pixDiscount = Setting::get('pix_discount_percentage', 0);
        $cart['pix_total'] = $cart['total'] * (1 - ($pixDiscount / 100));
    }

    /**
     * Página de checkout
     */
    public function index()
    {
        $cart = $this->getCart();
        $this->calculateTotals($cart);

        if (empty($cart['items'])) {
            return redirect()->route('cart.index')->with('error', 'Seu carrinho está vazio!');
        }

        $user = Auth::user();
        $asaasPixEnabled = $this->asaasService->isPixEnabled();
        $asaasCreditCardEnabled = $this->asaasService->isCreditCardEnabled();
        $asaasBoletoEnabled = $this->asaasService->isBoletoEnabled();

        $maxInstallments = Setting::get('max_installments', 12);
        $minInstallmentValue = Setting::get('min_installment_value', 10);
        $pixDiscount = Setting::get('pix_discount_percentage', 0);

        // Calcular parcelas disponíveis
        $availableInstallments = [];
        for ($i = 1; $i <= $maxInstallments; $i++) {
            $installmentValue = $cart['total'] / $i;
            if ($installmentValue >= $minInstallmentValue) {
                $availableInstallments[$i] = number_format($installmentValue, 2, ',', '.');
            }
        }

        return view('checkout.index', compact(
            'cart',
            'user',
            'asaasPixEnabled',
            'asaasCreditCardEnabled',
            'asaasBoletoEnabled',
            'availableInstallments',
            'maxInstallments',
            'pixDiscount'
        ));
    }

    /**
     * Processar pedido
     */
    public function process(Request $request)
    {
        $request->validate([
            // Dados pessoais
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255',
            'phone' => 'required|string|max:20',
            'cpf' => 'nullable|string|max:20',
            // Dados de entrega
            'zipcode' => 'required|string|max:10',
            'address' => 'required|string|max:255',
            'number' => 'required|string|max:20',
            'complement' => 'nullable|string|max:255',
            'district' => 'required|string|max:100',
            'city' => 'required|string|max:100',
            'state' => 'required|string|max:2',
            // Método de pagamento
            'payment_method' => 'required|in:pix,credit_card,boleto',
        ]);

        $cart = $this->getCart();
        $this->calculateTotals($cart);

        if (empty($cart['items'])) {
            return redirect()->route('cart.index')->with('error', 'Seu carrinho está vazio!');
        }

        // Validações adicionais para cartão de crédito
        if ($request->payment_method === 'credit_card') {
            $request->validate([
                'card_number' => 'required|string',
                'card_holder' => 'required|string|max:100',
                'card_expiration_month' => 'required|integer|min:1|max:12',
                'card_expiration_year' => 'required|integer|min:' . date('Y'),
                'card_cvv' => 'required|string|min:3|max:4',
                'installments' => 'required|integer|min:1',
            ]);
        }

        DB::beginTransaction();

        try {
            // Criar pedido
            $order = Order::create([
                'user_id' => Auth::id(),
                'order_number' => 'ORD-' . strtoupper(uniqid()),
                'customer_name' => $request->name,
                'customer_email' => $request->email,
                'customer_phone' => $request->phone,
                'customer_cpf' => $request->cpf,

                // Endereço de entrega
                'shipping_zipcode' => $request->zipcode,
                'shipping_address' => $request->address,
                'shipping_number' => $request->number,
                'shipping_complement' => $request->complement,
                'shipping_district' => $request->district,
                'shipping_city' => $request->city,
                'shipping_state' => $request->state,

                // Valores
                'subtotal' => $cart['subtotal'],
                'discount' => $cart['discount'],
                'shipping_cost' => $cart['shipping_cost'],
                'total' => $request->payment_method === 'pix' ? $cart['pix_total'] : $cart['total'],

                // Cupom
                'coupon_code' => $cart['coupon']['code'] ?? null,
                'coupon_discount' => $cart['discount'],

                // Pagamento
                'payment_method' => $request->payment_method,
                'payment_status' => Order::PAYMENT_PENDING,
                'status' => Order::STATUS_PENDING,

                // Parcelas (cartão)
                'installments' => $request->payment_method === 'credit_card' ? $request->installments : null,
                'installment_value' => $request->payment_method === 'credit_card' ? $cart['total'] / $request->installments : null,
            ]);

            // Adicionar itens ao pedido
            foreach ($cart['items'] as $item) {
                OrderItem::create([
                    'order_id' => $order->id,
                    'type' => $item['type'],
                    'product_id' => $item['type'] === 'product' ? $item['id'] : null,
                    'combo_id' => $item['type'] === 'combo' ? $item['id'] : null,
                    'name' => $item['name'],
                    'variant' => $item['variant'] ? json_encode($item['variant']) : null,
                    'price' => $item['price'],
                    'discount_price' => $item['discount_price'] ?? null,
                    'pix_price' => $item['pix_price'] ?? null,
                    'quantity' => $item['quantity'],
                    'subtotal' => ($item['discount_price'] ?? $item['price']) * $item['quantity'],
                ]);
            }

            // Processar pagamento no Asaas
            switch ($request->payment_method) {
                case 'pix':
                    $payment = $this->asaasService->createPixPayment($order);
                    $order->update([
                        'gateway_transaction_id' => $payment['id'],
                        'pix_qr_code' => $payment['encodedImage'] ?? null,
                        'pix_copy_paste' => $payment['payload'] ?? null,
                        'gateway_payment_link' => $payment['invoiceUrl'] ?? null,
                    ]);
                    break;

                case 'boleto':
                    $payment = $this->asaasService->createBoletoPayment($order);
                    $order->update([
                        'gateway_transaction_id' => $payment['id'],
                        'boleto_url' => $payment['invoiceUrl'] ?? null,
                        'boleto_barcode' => $payment['barcode'] ?? null,
                        'gateway_payment_link' => $payment['invoiceUrl'] ?? null,
                    ]);
                    break;

                case 'credit_card':
                    $cardData = [
                        'number' => $request->card_number,
                        'holder_name' => $request->card_holder,
                        'expiration_month' => $request->card_expiration_month,
                        'expiration_year' => $request->card_expiration_year,
                        'cvv' => $request->card_cvv,
                        'installments' => $request->installments,
                    ];
                    $payment = $this->asaasService->createCreditCardPayment($order, $cardData);

                    // Se pagamento com cartão foi confirmado
                    if (isset($payment['status']) && $payment['status'] === 'CONFIRMED') {
                        $order->update([
                            'payment_status' => Order::PAYMENT_PAID,
                            'status' => Order::STATUS_CONFIRMED,
                        ]);
                    }
                    break;
            }

            DB::commit();

            // Limpar carrinho
            Session::forget('cart');

            // Redirecionar para página de sucesso
            return redirect()->route('checkout.success', $order->order_number);

        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'Erro ao processar pedido: ' . $e->getMessage())->withInput();
        }
    }

    /**
     * Página de sucesso
     */
    public function success($orderNumber)
    {
        $order = Order::where('order_number', $orderNumber)->firstOrFail();

        // Verificar se o usuário é o dono do pedido
        if ($order->user_id && $order->user_id !== Auth::id()) {
            abort(403);
        }

        return view('checkout.success', compact('order'));
    }

    /**
     * API para consultar CEP
     */
    public function consultCep($cep)
    {
        $cep = preg_replace('/[^0-9]/', '', $cep);

        if (strlen($cep) !== 8) {
            return response()->json(['error' => 'CEP inválido'], 400);
        }

        $response = \Illuminate\Support\Facades\Http::get("https://viacep.com.br/ws/{$cep}/json/");

        if ($response->successful() && !$response->json('erro')) {
            return response()->json($response->json());
        }

        return response()->json(['error' => 'CEP não encontrado'], 404);
    }
}
