<?php

namespace App\Http\Controllers;

use App\Models\Product;
use App\Models\Combo;
use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Session;

class CartController extends Controller
{
    /**
     * Obter o carrinho da sessão
     */
    protected function getCart()
    {
        return Session::get('cart', [
            'items' => [],
            'coupon' => null,
        ]);
    }

    /**
     * Salvar o carrinho na sessão
     */
    protected function saveCart(array $cart): void
    {
        Session::put('cart', $cart);
    }

    /**
     * Adicionar produto ao carrinho
     */
    public function add(Request $request)
    {
        $request->validate([
            'type' => 'required|in:product,combo',
            'id' => 'required',
            'quantity' => 'required|integer|min:1',
            'variant' => 'nullable|array',
        ]);

        $cart = $this->getCart();

        if ($request->type === 'product') {
            $item = Product::with('images')->findOrFail($request->id);
        } else {
            $item = Combo::findOrFail($request->id);
        }

        // Gerar chave única para o item
        $itemKey = $request->type . '_' . $request->id;
        if ($request->filled('variant')) {
            $itemKey .= '_' . md5(json_encode($request->variant));
        }

        // Verificar se o item já existe no carrinho
        $existingIndex = collect($cart['items'])->search(function ($cartItem) use ($itemKey) {
            return $cartItem['key'] === $itemKey;
        });

        if ($existingIndex !== false) {
            // Atualizar quantidade
            $cart['items'][$existingIndex]['quantity'] += $request->quantity;
        } else {
            // Adicionar novo item
            $cart['items'][] = [
                'key' => $itemKey,
                'type' => $request->type,
                'id' => $request->id,
                'name' => $item->name,
                'image' => $request->type === 'product' && $item->images->where('is_main', true)->first()
                    ? $item->images->where('is_main', true)->first()->image_path
                    : ($item->image ?? null),
                'price' => $item->price,
                'discount_price' => $item->discount_price ?? null,
                'pix_price' => $item->pix_price ?? null,
                'quantity' => $request->quantity,
                'variant' => $request->variant ?? null,
            ];
        }

        $this->saveCart($cart);

        return redirect()->back()->with('success', 'Produto adicionado ao carrinho!');
    }

    /**
     * Atualizar quantidade de um item
     */
    public function update(Request $request)
    {
        $request->validate([
            'index' => 'required|integer',
            'quantity' => 'required|integer|min:1',
        ]);

        $cart = $this->getCart();

        if (isset($cart['items'][$request->index])) {
            $cart['items'][$request->index]['quantity'] = $request->quantity;
            $this->saveCart($cart);
        }

        return redirect()->back()->with('success', 'Carrinho atualizado!');
    }

    /**
     * Remover item do carrinho
     */
    public function remove(Request $request)
    {
        $request->validate([
            'index' => 'required|integer',
        ]);

        $cart = $this->getCart();

        if (isset($cart['items'][$request->index])) {
            array_splice($cart['items'], $request->index, 1);
            $this->saveCart($cart);
        }

        return redirect()->back()->with('success', 'Item removido do carrinho!');
    }

    /**
     * Limpar carrinho
     */
    public function clear()
    {
        Session::forget('cart');
        return redirect()->back()->with('success', 'Carrinho limpo!');
    }

    /**
     * Obter carrinho (API)
     */
    public function get()
    {
        $cart = $this->getCart();
        $this->calculateTotals($cart);

        return response()->json($cart);
    }

    /**
     * Calcular totais do carrinho
     */
    protected function calculateTotals(array &$cart): void
    {
        $cart['subtotal'] = collect($cart['items'])->sum(function ($item) {
            return ($item['discount_price'] ?? $item['price']) * $item['quantity'];
        });

        // Desconto do cupom
        $cart['discount'] = 0;
        if ($cart['coupon']) {
            // TODO: Calcular desconto do cupom
        }

        // Frete
        $cart['shipping_cost'] = 0;
        $cart['free_shipping'] = false;

        $freeShippingThreshold = Setting::get('free_shipping_threshold');
        if ($freeShippingThreshold && $cart['subtotal'] >= $freeShippingThreshold) {
            $cart['free_shipping'] = true;
        } else {
            $cart['shipping_cost'] = Setting::get('default_shipping_cost', 0);
        }

        // Total
        $cart['total'] = $cart['subtotal'] - $cart['discount'] + $cart['shipping_cost'];

        // Total PIX (com desconto adicional)
        $pixDiscount = Setting::get('pix_discount_percentage', 0);
        $cart['pix_total'] = $cart['total'] * (1 - ($pixDiscount / 100));
    }

    /**
     * Aplicar cupom de desconto
     */
    public function applyCoupon(Request $request)
    {
        $request->validate([
            'code' => 'required|string',
        ]);

        $coupon = \App\Models\Coupon::where('code', strtoupper($request->code))
            ->active()
            ->valid()
            ->first();

        if (!$coupon) {
            return redirect()->back()->with('error', 'Cupom inválido ou expirado!');
        }

        $cart = $this->getCart();
        $this->calculateTotals($cart);

        if ($coupon->min_purchase && $cart['subtotal'] < $coupon->min_purchase) {
            return redirect()->back()->with('error', 'Valor mínimo não atingido para este cupom!');
        }

        $cart['coupon'] = [
            'code' => $coupon->code,
            'type' => $coupon->type,
            'value' => $coupon->value,
        ];

        $this->saveCart($cart);

        return redirect()->back()->with('success', 'Cupom aplicado com sucesso!');
    }

    /**
     * Remover cupom
     */
    public function removeCoupon()
    {
        $cart = $this->getCart();
        $cart['coupon'] = null;
        $this->saveCart($cart);

        return redirect()->back()->with('success', 'Cupom removido!');
    }

    /**
     * Página do carrinho
     */
    public function index()
    {
        $cart = $this->getCart();
        $this->calculateTotals($cart);

        return view('cart.index', compact('cart'));
    }
}
