<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Variant;
use Illuminate\Http\Request;
use Illuminate\View\View;
use Illuminate\Http\RedirectResponse;

class VariantController extends Controller
{
    /**
     * Listar variantes
     */
    public function index(): View
    {
        $variants = Variant::with('product')->latest()->paginate(20);

        return view('admin.variants.index', compact('variants'));
    }

    /**
     * Mostrar formulário de criação
     */
    public function create(): View
    {
        $products = \App\Models\Product::all();
        return view('admin.variants.create', compact('products'));
    }

    /**
     * Salvar nova variante
     */
    public function store(Request $request): RedirectResponse
    {
        $request->validate([
            'product_id' => 'required|exists:products,id',
            'name' => 'required|string|max:255',
            'sku' => 'nullable|string|max:100|unique:variants,sku',
            'price' => 'required|numeric|min:0',
            'stock' => 'required|integer|min:0',
            'is_active' => 'boolean',
        ]);

        Variant::create([
            'product_id' => $request->product_id,
            'name' => $request->name,
            'sku' => $request->sku,
            'price' => $request->price,
            'stock' => $request->stock,
            'is_active' => $request->is_active ?? true,
        ]);

        return redirect()->route('admin.variants.index')->with('success', 'Variante criada com sucesso!');
    }

    /**
     * Mostrar detalhes da variante
     */
    public function show(Variant $variant): View
    {
        return view('admin.variants.show', compact('variant'));
    }

    /**
     * Mostrar formulário de edição
     */
    public function edit(Variant $variant): View
    {
        $products = \App\Models\Product::all();
        return view('admin.variants.edit', compact('variant', 'products'));
    }

    /**
     * Atualizar variante
     */
    public function update(Request $request, Variant $variant): RedirectResponse
    {
        $request->validate([
            'product_id' => 'required|exists:products,id',
            'name' => 'required|string|max:255',
            'sku' => 'nullable|string|max:100|unique:variants,sku,' . $variant->id,
            'price' => 'required|numeric|min:0',
            'stock' => 'required|integer|min:0',
            'is_active' => 'boolean',
        ]);

        $variant->update([
            'product_id' => $request->product_id,
            'name' => $request->name,
            'sku' => $request->sku,
            'price' => $request->price,
            'stock' => $request->stock,
            'is_active' => $request->is_active ?? true,
        ]);

        return redirect()->route('admin.variants.index')->with('success', 'Variante atualizada com sucesso!');
    }

    /**
     * Excluir variante
     */
    public function destroy(Variant $variant): RedirectResponse
    {
        $variant->delete();

        return redirect()->route('admin.variants.index')->with('success', 'Variante excluída com sucesso!');
    }

    /**
     * Toggle ativo/inativo
     */
    public function toggleActive(Variant $variant): RedirectResponse
    {
        $variant->update([
            'is_active' => !$variant->is_active,
        ]);

        return redirect()->back()->with('success', 'Status da variante atualizado!');
    }
}
