<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class SettingController extends Controller
{
    public function index()
    {
        $settings = Setting::all()->pluck('value', 'key')->toArray();

        return view('admin.settings.index', compact('settings'));
    }

    public function update(Request $request)
    {
        $validated = $request->validate([
            // Loja
            'store_name' => 'nullable|string|max:255',
            'store_email' => 'nullable|email|max:255',
            'store_phone' => 'nullable|string|max:20',
            'store_description' => 'nullable|string',

            // Asaas
            'asaas_api_key' => 'nullable|string|max:255',
            'asaas_sandbox' => 'nullable|boolean',
            'asaas_pix_enabled' => 'nullable|boolean',
            'asaas_credit_card_enabled' => 'nullable|boolean',
            'asaas_boleto_enabled' => 'nullable|boolean',

            // Mercado Pago
            'mercadopago_access_token' => 'nullable|string|max:500',
            'mercadopago_sandbox' => 'nullable|boolean',

            // Frete
            'free_shipping_threshold' => 'nullable|numeric|min:0',
            'default_shipping_cost' => 'nullable|numeric|min:0',

            // Parcelamento
            'max_installments' => 'nullable|integer|min:1|max:12',
            'min_installment_value' => 'nullable|numeric|min:0',

            // PIX
            'pix_discount_percentage' => 'nullable|numeric|min:0|max:100',
        ]);

        // Salvar configurações da loja
        foreach (['store_name', 'store_email', 'store_phone', 'store_description'] as $key) {
            if (isset($validated[$key])) {
                Setting::set($key, $validated[$key], 'text', 'general');
            }
        }

        // Salvar configurações do Asaas
        Setting::set('asaas_api_key', $validated['asaas_api_key'] ?? '', 'text', 'payment');
        Setting::set('asaas_sandbox', $request->has('asaas_sandbox') ? '1' : '0', 'boolean', 'payment');
        Setting::set('asaas_pix_enabled', $request->has('asaas_pix_enabled') ? '1' : '0', 'boolean', 'payment');
        Setting::set('asaas_credit_card_enabled', $request->has('asaas_credit_card_enabled') ? '1' : '0', 'boolean', 'payment');
        Setting::set('asaas_boleto_enabled', $request->has('asaas_boleto_enabled') ? '1' : '0', 'boolean', 'payment');

        // Salvar configurações do Mercado Pago
        Setting::set('mercadopago_access_token', $validated['mercadopago_access_token'] ?? '', 'text', 'payment');
        Setting::set('mercadopago_sandbox', $request->has('mercadopago_sandbox') ? '1' : '0', 'boolean', 'payment');

        // Salvar configurações de frete
        Setting::set('free_shipping_threshold', $validated['free_shipping_threshold'] ?? null, 'number', 'shipping');
        Setting::set('default_shipping_cost', $validated['default_shipping_cost'] ?? 0, 'number', 'shipping');

        // Salvar configurações de parcelamento
        Setting::set('max_installments', $validated['max_installments'] ?? 12, 'number', 'payment');
        Setting::set('min_installment_value', $validated['min_installment_value'] ?? 10, 'number', 'payment');

        // Salvar configurações de PIX
        Setting::set('pix_discount_percentage', $validated['pix_discount_percentage'] ?? 0, 'number', 'payment');

        return redirect()
            ->route('admin.settings.index')
            ->with('success', 'Configurações atualizadas com sucesso!');
    }

    public function testAsaasConnection(Request $request)
    {
        $request->validate([
            'api_key' => 'required|string',
            'sandbox' => 'nullable|boolean',
        ]);

        try {
            $client = new \GuzzleHttp\Client([
                'base_uri' => $request->sandbox ? 'https://sandbox.asaas.com/api/v3/' : 'https://www.asaas.com/api/v3/',
                'headers' => [
                    'access_token' => $request->api_key,
                    'Content-Type' => 'application/json',
                ],
                'timeout' => 10,
            ]);

            $response = $client->get('customers');
            $statusCode = $response->getStatusCode();

            if ($statusCode === 200) {
                return response()->json([
                    'success' => true,
                    'message' => 'Conexão com Asaas estabelecida com sucesso!',
                ]);
            }

            return response()->json([
                'success' => false,
                'message' => 'Erro ao conectar com Asaas. Status: ' . $statusCode,
            ], 400);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Erro de conexão: ' . $e->getMessage(),
            ], 400);
        }
    }
}
