<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Models\Category;
use App\Models\ProductImage;
use App\Models\ProductVariant;
use App\Models\Variant;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;

class ProductController extends Controller
{
    public function index(Request $request)
    {
        $query = Product::with('category', 'images');

        // Busca
        if ($request->search) {
            $query->where('name', 'like', '%' . $request->search . '%')
                ->orWhere('sku', 'like', '%' . $request->search . '%');
        }

        // Filtro por categoria
        if ($request->category) {
            $query->where('category_id', $request->category);
        }

        // Filtro por status
        if ($request->status === 'active') {
            $query->where('is_active', true);
        } elseif ($request->status === 'inactive') {
            $query->where('is_active', false);
        }

        // Filtro por destaque
        if ($request->featured === 'yes') {
            $query->where('is_featured', true);
        } elseif ($request->featured === 'no') {
            $query->where('is_featured', false);
        }

        $products = $query->latest()->paginate(20);
        $categories = Category::all();

        return view('admin.products.index', compact('products', 'categories'));
    }

    public function create()
    {
        $categories = Category::orderBy('name')->get();
        $variants = Variant::active()->orderBy('sort_order')->get()->groupBy('type');

        return view('admin.products.create', compact('categories', 'variants'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'sku' => 'nullable|string|max:100|unique:products,sku',
            'slug' => 'nullable|string|max:255|unique:products,slug',
            'short_description' => 'nullable|string|max:500',
            'description' => 'nullable|string',
            'category_id' => 'nullable|exists:categories,id',
            'brand' => 'nullable|string|max:100',
            'weight' => 'nullable|numeric',
            'flavor' => 'nullable|string|max:100',
            'servings' => 'nullable|integer',
            'ingredients' => 'nullable|string',
            'usage_instructions' => 'nullable|string',
            'warnings' => 'nullable|string',
            'price' => 'required|numeric|min:0',
            'discount_price' => 'nullable|numeric|min:0|lt:price',
            'stock' => 'nullable|integer|min:0',
            'is_featured' => 'nullable|boolean',
            'is_active' => 'nullable|boolean',
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string|max:500',
            'video_url' => 'nullable|string|max:500',
            'video_type' => 'nullable|in:youtube,vimeo,upload',
            'images' => 'nullable|array|max:6',
            'images.*' => 'nullable|image|mimes:jpg,jpeg,png,webp|max:5120',
            'variant_data' => 'nullable|array',
        ]);

        // Gerar slug se não fornecido
        if (empty($validated['slug'])) {
            $validated['slug'] = Str::slug($validated['name']);
        }

        // Calcular porcentagem de desconto
        if (!empty($validated['discount_price'])) {
            $validated['discount_percentage'] = round((($validated['price'] - $validated['discount_price']) / $validated['price']) * 100);
        }

        // Configurar valores padrão
        $validated['is_featured'] = $request->has('is_featured');
        $validated['is_active'] = $request->has('is_active') ? true : false;
        $validated['max_images'] = 6;

        $product = Product::create($validated);

        // Upload de imagens
        if ($request->hasFile('images')) {
            foreach ($request->file('images') as $index => $image) {
                $path = $image->store('products', 'public');
                ProductImage::create([
                    'product_id' => $product->id,
                    'image_path' => $path,
                    'is_main' => $index === 0,
                    'order' => $index,
                ]);
            }
        }

        // Salvar variantes
        if ($request->filled('variant_data')) {
            foreach ($request->variant_data as $variantId => $values) {
                $variant = Variant::find($variantId);
                if ($variant) {
                    foreach ($values as $valueData) {
                        if (!empty($valueData['value'])) {
                            ProductVariant::create([
                                'product_id' => $product->id,
                                'variant_id' => $variantId,
                                'value' => $valueData['value'],
                                'price_adjustment' => $valueData['price_adjustment'] ?? 0,
                                'sku' => $valueData['sku'] ?? null,
                                'stock' => $valueData['stock'] ?? 0,
                                'is_active' => true,
                            ]);
                        }
                    }
                }
            }
        }

        return redirect()
            ->route('admin.products.index')
            ->with('success', 'Produto criado com sucesso!');
    }

    public function show(Product $product)
    {
        $product->load(['category', 'images', 'variants', 'productVariants']);
        return view('admin.products.show', compact('product'));
    }

    public function edit(Product $product)
    {
        $product->load(['images', 'variants', 'productVariants']);
        $categories = Category::orderBy('name')->get();
        $variants = Variant::active()->orderBy('sort_order')->get()->groupBy('type');

        return view('admin.products.edit', compact('product', 'categories', 'variants'));
    }

    public function update(Request $request, Product $product)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'sku' => 'nullable|string|max:100|unique:products,sku,' . $product->id,
            'slug' => 'nullable|string|max:255|unique:products,slug,' . $product->id,
            'short_description' => 'nullable|string|max:500',
            'description' => 'nullable|string',
            'category_id' => 'nullable|exists:categories,id',
            'brand' => 'nullable|string|max:100',
            'weight' => 'nullable|numeric',
            'flavor' => 'nullable|string|max:100',
            'servings' => 'nullable|integer',
            'ingredients' => 'nullable|string',
            'usage_instructions' => 'nullable|string',
            'warnings' => 'nullable|string',
            'price' => 'required|numeric|min:0',
            'discount_price' => 'nullable|numeric|min:0|lt:price',
            'stock' => 'nullable|integer|min:0',
            'is_featured' => 'nullable|boolean',
            'is_active' => 'nullable|boolean',
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string|max:500',
            'video_url' => 'nullable|string|max:500',
            'video_type' => 'nullable|in:youtube,vimeo,upload',
            'images' => 'nullable|array|max:6',
            'images.*' => 'nullable|image|mimes:jpg,jpeg,png,webp|max:5120',
            'delete_images' => 'nullable|array',
            'delete_images.*' => 'exists:product_images,id',
            'variant_data' => 'nullable|array',
        ]);

        // Gerar slug se não fornecido
        if (empty($validated['slug'])) {
            $validated['slug'] = Str::slug($validated['name']);
        }

        // Calcular porcentagem de desconto
        if (!empty($validated['discount_price'])) {
            $validated['discount_percentage'] = round((($validated['price'] - $validated['discount_price']) / $validated['price']) * 100);
        } else {
            $validated['discount_percentage'] = null;
        }

        // Configurar valores
        $validated['is_featured'] = $request->has('is_featured');
        $validated['is_active'] = $request->has('is_active');

        $product->update($validated);

        // Deletar imagens marcadas
        if ($request->filled('delete_images')) {
            foreach ($request->delete_images as $imageId) {
                $image = ProductImage::find($imageId);
                if ($image && $image->product_id === $product->id) {
                    Storage::disk('public')->delete($image->image_path);
                    $image->delete();
                }
            }
        }

        // Upload de novas imagens
        if ($request->hasFile('images')) {
            $currentOrder = $product->images()->max('order') ?? 0;
            foreach ($request->file('images') as $index => $image) {
                $path = $image->store('products', 'public');
                ProductImage::create([
                    'product_id' => $product->id,
                    'image_path' => $path,
                    'is_main' => $product->images()->count() === 0 && $index === 0,
                    'order' => $currentOrder + $index + 1,
                ]);
            }
        }

        // Atualizar variantes
        $product->productVariants()->delete();
        if ($request->filled('variant_data')) {
            foreach ($request->variant_data as $variantId => $values) {
                $variant = Variant::find($variantId);
                if ($variant) {
                    foreach ($values as $valueData) {
                        if (!empty($valueData['value'])) {
                            ProductVariant::create([
                                'product_id' => $product->id,
                                'variant_id' => $variantId,
                                'value' => $valueData['value'],
                                'price_adjustment' => $valueData['price_adjustment'] ?? 0,
                                'sku' => $valueData['sku'] ?? null,
                                'stock' => $valueData['stock'] ?? 0,
                                'is_active' => true,
                            ]);
                        }
                    }
                }
            }
        }

        return redirect()
            ->route('admin.products.index')
            ->with('success', 'Produto atualizado com sucesso!');
    }

    public function destroy(Product $product)
    {
        // Deletar imagens
        foreach ($product->images as $image) {
            Storage::disk('public')->delete($image->image_path);
        }

        // Deletar variantes do produto
        $product->productVariants()->delete();

        // Deletar produto
        $product->delete();

        return redirect()
            ->route('admin.products.index')
            ->with('success', 'Produto deletado com sucesso!');
    }

    public function toggleActive(Product $product)
    {
        $product->update(['is_active' => !$product->is_active]);
        return back()->with('success', 'Status do produto atualizado!');
    }

    public function toggleFeatured(Product $product)
    {
        $product->update(['is_featured' => !$product->is_featured]);
        return back()->with('success', 'Status de destaque atualizado!');
    }

    public function setMainImage(Product $product, ProductImage $image)
    {
        if ($image->product_id !== $product->id) {
            return back()->with('error', 'Imagem não pertence a este produto.');
        }

        // Remover destaque de todas as imagens
        $product->images()->update(['is_main' => false]);

        // Definir nova imagem principal
        $image->update(['is_main' => true]);

        return back()->with('success', 'Imagem principal definida!');
    }

    /**
     * Mostrar formulário de importação
     */
    public function import()
    {
        return view('admin.products.import');
    }

    /**
     * Processar importação de CSV
     */
    public function processImport(Request $request)
    {
        $request->validate([
            'csv_file' => 'required|mimes:csv,txt|max:10240', // max 10MB
        ]);

        $file = $request->file('csv_file');
        $path = $file->getRealPath();

        $imported = 0;
        $errors = [];
        $row = 0;

        if (($handle = fopen($path, 'r')) !== false) {
            // Get headers
            $headers = fgetcsv($handle, 0, ',');

            // Expected columns: name, sku, category, price, discount_price, stock, description, is_featured, is_active
            while (($data = fgetcsv($handle, 0, ',')) !== false) {
                $row++;

                // Skip empty rows
                if (empty(array_filter($data))) {
                    continue;
                }

                try {
                    $productData = [
                        'name' => $data[0] ?? null,
                        'sku' => $data[1] ?? null,
                        'price' => (float) ($data[3] ?? 0),
                        'discount_price' => !empty($data[4]) ? (float) $data[4] : null,
                        'stock' => (int) ($data[5] ?? 0),
                        'description' => $data[6] ?? null,
                        'is_featured' => !empty($data[7]) ? filter_var($data[7], FILTER_VALIDATE_BOOLEAN) : false,
                        'is_active' => !empty($data[8]) ? filter_var($data[8], FILTER_VALIDATE_BOOLEAN) : true,
                    ];

                    // Generate slug
                    if ($productData['name']) {
                        $productData['slug'] = Str::slug($productData['name']);

                        // Check for duplicate slug
                        $count = Product::where('slug', 'LIKE', $productData['slug'] . '%')->count();
                        if ($count > 0) {
                            $productData['slug'] = $productData['slug'] . '-' . ($count + 1);
                        }
                    }

                    // Find category
                    if (!empty($data[2])) {
                        $category = Category::where('name', 'like', '%' . $data[2] . '%')->first();
                        if ($category) {
                            $productData['category_id'] = $category->id;
                        }
                    }

                    // Calculate discount percentage
                    if ($productData['discount_price']) {
                        $productData['discount_percentage'] = round((($productData['price'] - $productData['discount_price']) / $productData['price']) * 100);
                    }

                    $productData['max_images'] = 6;

                    // Create or update product
                    if ($productData['name']) {
                        Product::updateOrCreate(
                            ['sku' => $productData['sku']],
                            $productData
                        );
                        $imported++;
                    }
                } catch (\Exception $e) {
                    $errors[] = "Linha $row: " . $e->getMessage();
                }
            }

            fclose($handle);
        }

        return redirect()
            ->route('admin.products.index')
            ->with('success', "Importação concluída! {$imported} produtos importados com sucesso.")
            ->with('errors', $errors);
    }

    /**
     * Download template de CSV
     */
    public function downloadTemplate()
    {
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename=produtos_template.csv',
        ];

        $callback = function() {
            $file = fopen('php://output', 'w');
            fputcsv($file, [
                'name',
                'sku',
                'category',
                'price',
                'discount_price',
                'stock',
                'description',
                'is_featured',
                'is_active'
            ]);

            // Exemplo de linha
            fputcsv($file, [
                'Colágeno Verisol + Vitamina C',
                'COLAGEN-001',
                'Saúde & Beleza',
                '89.90',
                '69.90',
                '100',
                'Colágeno hidrolisado com vitamina C para pele mais firme e saudável.',
                '1',
                '1'
            ]);

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }
}
