<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Order;
use Illuminate\Http\Request;
use Illuminate\View\View;
use Illuminate\Http\RedirectResponse;

class OrderController extends Controller
{
    /**
     * Listar pedidos
     */
    public function index(Request $request): View
    {
        $query = Order::with('items', 'user')->latest();

        // Filtros
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('payment_status')) {
            $query->where('payment_status', $request->payment_status);
        }

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('order_number', 'like', "%{$search}%")
                  ->orWhere('customer_name', 'like', "%{$search}%")
                  ->orWhere('customer_email', 'like', "%{$search}%");
            });
        }

        $orders = $query->paginate(20);

        $stats = [
            'total' => Order::count(),
            'pending' => Order::where('status', Order::STATUS_PENDING)->count(),
            'confirmed' => Order::where('status', Order::STATUS_CONFIRMED)->count(),
            'processing' => Order::where('status', Order::STATUS_PROCESSING)->count(),
            'shipped' => Order::where('status', Order::STATUS_SHIPPED)->count(),
            'delivered' => Order::where('status', Order::STATUS_DELIVERED)->count(),
            'cancelled' => Order::where('status', Order::STATUS_CANCELLED)->count(),
        ];

        return view('admin.orders.index', compact('orders', 'stats'));
    }

    /**
     * Mostrar detalhes do pedido
     */
    public function show(Order $order): View
    {
        $order->load('items.product', 'items.combo', 'user');

        return view('admin.orders.show', compact('order'));
    }

    /**
     * Atualizar status do pedido
     */
    public function update(Request $request, Order $order): RedirectResponse
    {
        $request->validate([
            'status' => 'required|in:' . implode(',', [
                Order::STATUS_PENDING,
                Order::STATUS_CONFIRMED,
                Order::STATUS_PROCESSING,
                Order::STATUS_SHIPPED,
                Order::STATUS_DELIVERED,
                Order::STATUS_CANCELLED,
            ]),
            'tracking_code' => 'nullable|string|max:255',
            'notes' => 'nullable|string|max:1000',
        ]);

        $order->update([
            'status' => $request->status,
            'tracking_code' => $request->tracking_code,
            'admin_notes' => $request->notes,
        ]);

        return redirect()->back()->with('success', 'Status do pedido atualizado!');
    }

    /**
     * Atualizar status de pagamento
     */
    public function updatePaymentStatus(Request $request, Order $order): RedirectResponse
    {
        $request->validate([
            'payment_status' => 'required|in:' . implode(',', [
                Order::PAYMENT_PENDING,
                Order::PAYMENT_PAID,
                Order::PAYMENT_CANCELLED,
                Order::PAYMENT_REFUNDED,
            ]),
        ]);

        $order->update([
            'payment_status' => $request->payment_status,
        ]);

        // Se pagamento foi confirmado, atualizar status do pedido
        if ($request->payment_status === Order::PAYMENT_PAID && $order->status === Order::STATUS_PENDING) {
            $order->update(['status' => Order::STATUS_CONFIRMED]);
        }

        // Se pagamento foi cancelado, cancelar pedido
        if ($request->payment_status === Order::PAYMENT_CANCELLED) {
            $order->update(['status' => Order::STATUS_CANCELLED]);
        }

        return redirect()->back()->with('success', 'Status de pagamento atualizado!');
    }

    /**
     * Exportar pedido (PDF/Imprimir)
     */
    public function print(Order $order): View
    {
        $order->load('items.product', 'items.combo', 'user');

        return view('admin.orders.print', compact('order'));
    }
}
