<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Combo;
use App\Models\ComboItem;
use App\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class ComboController extends Controller
{
    public function index(Request $request)
    {
        $query = Combo::with('items.product');

        // Busca
        if ($request->search) {
            $query->where('name', 'like', '%' . $request->search . '%');
        }

        // Filtro por status
        if ($request->status === 'active') {
            $query->where('is_active', true);
        } elseif ($request->status === 'inactive') {
            $query->where('is_active', false);
        }

        $combos = $query->latest()->paginate(20);

        return view('admin.combos.index', compact('combos'));
    }

    public function create()
    {
        $products = Product::active()->orderBy('name')->get();
        return view('admin.combos.create', compact('products'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'slug' => 'nullable|string|max:255|unique:combos,slug',
            'description' => 'nullable|string',
            'original_price' => 'required|numeric|min:0',
            'price' => 'required|numeric|min:0|lt:original_price',
            'pix_price' => 'nullable|numeric|min:0|lt:price',
            'max_installments' => 'nullable|integer|min:1|max:12',
            'min_installment_value' => 'nullable|numeric|min:0',
            'is_active' => 'nullable|boolean',
            'is_featured' => 'nullable|boolean',
            'free_shipping_threshold' => 'nullable|integer|min:0',
            'image' => 'nullable|image|mimes:jpg,jpeg,png,webp|max:5120',
            'video_url' => 'nullable|string|max:500',
            'gifts' => 'nullable|array',
            'products' => 'required|array|min:1',
            'products.*.id' => 'required|exists:products,id',
            'products.*.quantity' => 'required|integer|min:1',
        ]);

        // Gerar slug se não fornecido
        if (empty($validated['slug'])) {
            $validated['slug'] = Str::slug($validated['name']);
        }

        // Configurar valores
        $validated['is_active'] = $request->has('is_active');
        $validated['is_featured'] = $request->has('is_featured');
        $validated['gifts'] = $request->gifts ?? [];

        // Upload de imagem
        if ($request->hasFile('image')) {
            $validated['image'] = $request->file('image')->store('combos', 'public');
        }

        $combo = Combo::create($validated);

        // Adicionar produtos ao combo
        foreach ($request->products as $productData) {
            ComboItem::create([
                'combo_id' => $combo->id,
                'product_id' => $productData['id'],
                'quantity' => $productData['quantity'],
            ]);
        }

        return redirect()
            ->route('admin.combos.index')
            ->with('success', 'Combo criado com sucesso!');
    }

    public function show(Combo $combo)
    {
        $combo->load('items.product');
        return view('admin.combos.show', compact('combo'));
    }

    public function edit(Combo $combo)
    {
        $combo->load('items');
        $products = Product::active()->orderBy('name')->get();
        return view('admin.combos.edit', compact('combo', 'products'));
    }

    public function update(Request $request, Combo $combo)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'slug' => 'nullable|string|max:255|unique:combos,slug,' . $combo->id,
            'description' => 'nullable|string',
            'original_price' => 'required|numeric|min:0',
            'price' => 'required|numeric|min:0|lt:original_price',
            'pix_price' => 'nullable|numeric|min:0|lt:price',
            'max_installments' => 'nullable|integer|min:1|max:12',
            'min_installment_value' => 'nullable|numeric|min:0',
            'is_active' => 'nullable|boolean',
            'is_featured' => 'nullable|boolean',
            'free_shipping_threshold' => 'nullable|integer|min:0',
            'image' => 'nullable|image|mimes:jpg,jpeg,png,webp|max:5120',
            'video_url' => 'nullable|string|max:500',
            'delete_image' => 'nullable|boolean',
            'gifts' => 'nullable|array',
            'products' => 'required|array|min:1',
            'products.*.id' => 'required|exists:products,id',
            'products.*.quantity' => 'required|integer|min:1',
        ]);

        // Gerar slug se não fornecido
        if (empty($validated['slug'])) {
            $validated['slug'] = Str::slug($validated['name']);
        }

        // Configurar valores
        $validated['is_active'] = $request->has('is_active');
        $validated['is_featured'] = $request->has('is_featured');
        $validated['gifts'] = $request->gifts ?? [];

        // Deletar imagem antiga se marcado
        if ($request->has('delete_image') && $combo->image) {
            Storage::disk('public')->delete($combo->image);
            $validated['image'] = null;
        }

        // Upload de nova imagem
        if ($request->hasFile('image')) {
            if ($combo->image) {
                Storage::disk('public')->delete($combo->image);
            }
            $validated['image'] = $request->file('image')->store('combos', 'public');
        }

        $combo->update($validated);

        // Recriar itens do combo
        $combo->items()->delete();
        foreach ($request->products as $productData) {
            ComboItem::create([
                'combo_id' => $combo->id,
                'product_id' => $productData['id'],
                'quantity' => $productData['quantity'],
            ]);
        }

        return redirect()
            ->route('admin.combos.index')
            ->with('success', 'Combo atualizado com sucesso!');
    }

    public function destroy(Combo $combo)
    {
        // Deletar imagem
        if ($combo->image) {
            Storage::disk('public')->delete($combo->image);
        }

        // Deletar itens
        $combo->items()->delete();

        // Deletar combo
        $combo->delete();

        return redirect()
            ->route('admin.combos.index')
            ->with('success', 'Combo deletado com sucesso!');
    }

    public function toggleActive(Combo $combo)
    {
        $combo->update(['is_active' => !$combo->is_active]);
        return back()->with('success', 'Status do combo atualizado!');
    }

    public function toggleFeatured(Combo $combo)
    {
        $combo->update(['is_featured' => !$combo->is_featured]);
        return back()->with('success', 'Status de destaque atualizado!');
    }
}
